<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Word;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CategoryController extends Controller
{
    public function save(Request $request): \Illuminate\Http\JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string',
            'status' => 'nullable',
        ],[
            'title.required' => 'عنوان را وارد کنید',
            'title.string' => 'عنوان وارد شده معتبر نیست',
            'status.required' => 'وضعیت را انتخاب کنید',
            'status.in' => 'وضعیت انتخاب شده معتبر نیست',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->messages()->all(),
                'keys' => $validator->messages()->keys()
            ], 400);
        }
        if (Category::where('title', '=', request('title'))->exists()) {
            return response()->json([
                'errors' => [
                    'عنوان دسته بندی تکراری است.'
                ],
                'keys' => [
                    'title'
                ]
            ], 400);
        }
        $category = new Category();
        $category->title = request('title');
        $category->status = request('status') ?? "0";
        $category->image = request('image');
        $category->save();
        return response()->json( [
            'message' => 'دسته بندی ایجاد گردید',
            'category' => $category
        ], 200);
    }

    public function search(Request $request): \Illuminate\Http\JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string'
        ]);
        if ($validator->fails()) {
            return response()->json([
                'message' => $validator->getMessageBag()->first()
            ], 400);
        }
        $cats = Category::where('title', 'LIKE', '%' . request('title') . '%')->get();

        return response()->json([
            'search' => request('title'),
            'list' => $cats
        ], 200);
    }

    public function all(): \Illuminate\Http\JsonResponse
    {
        $cats = Category::all();
        return response()->json([
            'list' => $cats
        ], 200);
    }

    public function update(Request $request, $cat_id): \Illuminate\Http\JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string',
            'status' => 'nullable',
        ],[
            'title.required' => 'عنوان را وارد کنید',
            'title.string' => 'عنوان وارد شده معتبر نیست',
            'status.required' => 'وضعیت را انتخاب کنید',
            'status.in' => 'وضعیت انتخاب شده معتبر نیست',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->messages()->all(),
                'keys' => $validator->messages()->keys()
            ], 400);
        }

        if (!Category::where('_id', '=', $cat_id)->exists()) {
            return response()->json([
                'errors' => [
                    'عنوان دسته بندی تکراری است.'
                ],
                'keys' => [
                    'title'
                ]
            ], 400);
        }

        $cat = Category::where('_id', '=', request('cat_id'))->first();
        if (Category::where('title', '=', request('title'))->where('_id', '!=', $cat->id)->exists()) {
            return response()->json([
                'errors' => [
                    'عنوان دسته بندی تکراری است.'
                ],
                'keys' => [
                    'title'
                ]
            ], 400);
        }

        if ($cat->title == 'عمومی') {
            return response()->json([
                'errors' => [
                    'قادر به ویرایش دسته بندی عمومی نیستید.'
                ],
                'keys' => [
                    'title'
                ]
            ], 400);
        }
        $cat->title = request('title');
        $cat->status = request('status') ?? '0';
        $cat->save();

        return response()->json( [
            'message' => 'دسته بندی ویرایش شد',
            'category' => $cat
        ], 200);
    }

    public function getWithId($id): \Illuminate\Http\JsonResponse
    {
        $cat = Category::where('_id', $id)->first();
        if ($cat) {
            return response()->json([
                'id' => $id,
                'category' => $cat,
            ], 200);
        } else {
            return response()->json([
                'message' => 'invalid id',
            ], 400);
        }
    }
}
